const loginDetail = JSON.parse(sessionStorage.getItem("loginDetail"));
const username = loginDetail ? loginDetail.username : "";
const role = loginDetail ? loginDetail.role : "";

let allApplications = [];
let currentPage = 1;
const pageSize = 10;

// Utility: Get CSS class for status color
function getStatusClass(status) {
  switch (status.toLowerCase()) {
    case "approved":
      return "bg-success text-white";
    case "pre_processing":
      return "bg-info text-white";
    case "pre_approved":
    case "processing":
      return "bg-primary text-white";
    case "rejected":
      return "bg-danger text-white";
    case "new":
    default:
      return "bg-warning text-dark";
  }
}

function filterApplications() {
  renderPage(currentPage);
}

async function fetchApplicationStudent() {
  const { data, error } = await window.supabase
    .from("student_applications")
    .select(
      "id, register_by, full_name, status, remark, profile_photo, finalStatus, OfferLetterStatus, Passport_FPage"
    )
    .order("created_at", {
      ascending: false,
    });

  if (error) {
    console.error("Error fetching student applications:", error.message);
    return;
  }

  allApplications = data;
  renderPage(currentPage);

  const statusFilterElement = document.getElementById("statusFilter");
  const searchInputElement = document.getElementById("searchInput");

  if (statusFilterElement && !statusFilterElement._hasChangeListener) {
    statusFilterElement.addEventListener("change", filterApplications);
    statusFilterElement._hasChangeListener = true;
  }
  if (searchInputElement && !searchInputElement._hasInputListener) {
    searchInputElement.addEventListener("input", filterApplications);
    searchInputElement._hasInputListener = true;
  }
}

const tbody = document.querySelector("tbody");

// Attach a single click listener to tbody using event delegation
tbody.addEventListener("click", async (e) => {
  const target = e.target;

  if (target.classList.contains("update-remark")) {
    if (target.disabled) return;
    target.disabled = true;

    const id = target.dataset.id;
    const input = document.querySelector(`.remark-input[data-id="${id}"]`);
    const newRemark = input?.value ?? "";

    try {
      const { error } = await window.supabase
        .from("student_applications")
        .update({
          remark: newRemark,
        })
        .eq("id", id);

      if (error) {
        console.error("Error updating remark:", error.message);
        alert("Failed to update remark.");
      } else {
        alert("Remark updated successfully.");
        const updatedAppIndex = allApplications.findIndex(
          (app) => app.id === id
        );
        if (updatedAppIndex !== -1) {
          allApplications[updatedAppIndex].remark = newRemark;
          renderPage(currentPage);
        }
      }
    } catch (err) {
      console.error("Unexpected error:", err);
      alert("Unexpected error updating remark.");
    } finally {
      target.disabled = false;
    }
    return;
  }

  if (target.classList.contains("submit-btn")) {
    const id = target.dataset.id;
    await handleFileUpload(id);
    return;
  }

  if (target.classList.contains("submit-uploadZip")) {
    const id = target.dataset.id;
    await uploadZip(id);
    return;
  }

  if (target.classList.contains("full-pdf")) {
    const id = target.dataset.id;
    const student = await getStudentDetails(id);
    if (student) generatePDF(student);
    return;
  }

  if (target.classList.contains("image-student")) {
    const id = target.dataset.id;
    const photo = await getImageDetail(id);
    if (photo) await generateCombinedAttachmentsZip(photo);
    return;
  }

  if (target.classList.contains("image-receipt")) {
    const id = target.dataset.id;
    const photo = await getReceiptDetail(id);
    if (photo) generatePDFReceipt(photo);
    return;
  }

  if (target.classList.contains("passport-fpage")) {
    const id = target.dataset.id;
    try {
      await downloadPassportPDF(id);
    } catch (error) {
      console.error("Failed to download passport PDF:", error);
      alert("Failed to download passport PDF.");
    }
  }
});

// Attach a single CHANGE listener to tbody using event delegation for status-select
tbody.addEventListener("change", async (e) => {
  const target = e.target;
  if (target.classList.contains("status-select")) {
    const id = target.dataset.id;
    const newStatus = target.value;

    console.log("Attempting to update status for ID:", id, "to:", newStatus);

    const { error } = await window.supabase
      .from("student_applications")
      .update({
        status: newStatus,
      })
      .eq("id", id);

    if (error) {
      console.error("Error updating status:", error.message);
      alert("Failed to update status. Please check console for details.");
    } else {
      alert("Status updated to: " + newStatus);

      const updatedAppIndex = allApplications.findIndex((app) => app.id === id);
      if (updatedAppIndex !== -1) {
        allApplications[updatedAppIndex].status = newStatus;
      }
      renderPage(currentPage);
    }
  }
});

function renderPage(page) {
  const statusFilter = document
    .getElementById("statusFilter")
    .value.toLowerCase();
  const searchInput = document
    .getElementById("searchInput")
    .value.toLowerCase()
    .trim();

  const filteredApplications = allApplications.filter((application) => {
    const rowStatus = (application.status ?? "new").toLowerCase();
    const studentName = (application.full_name ?? "").toLowerCase();

    const statusMatch = statusFilter === "all" || rowStatus === statusFilter;
    const nameMatch = searchInput === "" || studentName.includes(searchInput);

    return statusMatch && nameMatch;
  });

  const start = (page - 1) * pageSize;
  const end = start + pageSize;
  const pageData = filteredApplications.slice(start, end);
  const tbody = document.querySelector("tbody");
  tbody.innerHTML = "";

  pageData.forEach((application, index) => {
    const continuousIndex = start + index + 1;
    const status = application.status ?? "new";
    const remark = application.remark ?? "";
    let downloadOtherLetter = "Letter's already submit.";
    let downloadOfferLetter = "Letter's already submit.";
    if (!application.finalStatus) {
      downloadOtherLetter = `<div id="upload-container-${application.id}" style="display: flex; flex-direction: column; gap: 10px; padding: 8px; max-width: 220px;">
            <div style="display: flex; flex-direction: column; gap: 6px;">
                <label for="upload2-${application.id}" style="font-size: 0.8rem;">EMGS</label>
                <input type="file" accept="image/*" id="upload2-${application.id}" class="form-control form-control-sm" />
            </div>
            <div style="display: flex; flex-direction: column; gap: 6px;">
                <label for="upload3-${application.id}" style="font-size: 0.8rem;">EVAL</label>
                <input type="file" accept="image/*" id="upload3-${application.id}" class="form-control form-control-sm" />
            </div>
            <button class="submit-btn btn btn-sm btn-success mt-2" type="button" data-id="${application.id}" style="align-self: start; padding: 4px 12px;">Submit</button>
        </div>`;
    }
    if (!application.OfferLetterStatus) {
      downloadOfferLetter = `<div id="upload-container-offer-${application.id}" style="display: flex; flex-direction: column; gap: 10px; padding: 8px; max-width: 220px;">
            <div style="display: flex; flex-direction: column; gap: 6px;">
                <label for="upload-zip-${application.id}" style="font-size: 0.8rem;">Offer Letter</label>
                <input type="file" accept=".zip" id="upload-zip-${application.id}" class="form-control form-control-sm" />
            </div>
            <button class="submit-uploadZip btn btn-sm btn-success mt-2" type="button" data-id="${application.id}" style="align-self: start; padding: 4px 12px;">Submit</button>
        </div>`;
    }

    const row = `
<tr data-id="${application.id}">
    <td class="text-sm text-center align-middle" style="width: 40px;">${continuousIndex}</td>
    <td class="align-middle" style="min-width: 180px;">
        <div class="d-flex flex-column">
            <span class="fw-semibold text-sm">${application.register_by}</span>
            <span class="text-muted small student-name" data-id="${
              application.id
            }">
                ${application.full_name}
            </span>
        </div>
    </td>
    <td class="align-middle" style="min-width: 140px;">
        <select class="form-select form-select-sm text-xs status-select ${getStatusClass(
          status
        )}" data-id="${application.id}">
            ${[
              "New",
              "Processing",
              "Pre_approved",
              "Pre_Processing",
              "Approved",
              "Rejected",
            ]
              .map(
                (s) =>
                  `<option value="${s}" ${
                    status.toLowerCase() === s.toLowerCase() ? "selected" : ""
                  }>${s.replace("_", " ")}</option>`
              )
              .join("")}
        </select>
    </td>
    <td class="align-middle" style="min-width: 200px;">
        <div class="d-flex flex-column gap-1">
            <textarea class="form-control form-control-sm remark-input" rows="2" style="resize: vertical;" data-id="${
              application.id
            }" placeholder="Enter remark">${remark}</textarea>
            <button class="btn btn-success btn-sm update-remark px-2 py-1 align-self-start" style="width: 60px; font-size: 0.75rem;" data-id="${
              application.id
            }">Update</button>
        </div>
    </td>
    <td class="text-center align-middle" style="width: 60px;">
        <i class="material-symbols-rounded text-black download-icon image-receipt" data-id="${
          application.id
        }" style="cursor:pointer;" title="View Receipt">receipt_long</i>
    </td>
    <td class="align-middle" style="min-width: 220px;">
        ${downloadOfferLetter}
    </td>
    <td class="align-middle" style="min-width: 220px;">
        ${downloadOtherLetter}
    </td>
    <td class="text-center align-middle" style="width: 140px; display: flex; flex-direction: column; gap: 4px;">
        <div style="display: flex; align-items: center; gap: 6px; cursor: pointer;" title="Download Application Form PDF">
            <i class="material-symbols-rounded text-black download-icon full-pdf me-2" data-id="${
              application.id
            }" style="cursor:pointer;" title="Download Full PDF">download</i>
            <span style="font-size: 0.75rem;">Application</span>
        </div>
        <div style="display: flex; align-items: center; gap: 6px; cursor: pointer;" title="View Attachment File">
            <i class="material-symbols-rounded text-black download-icon image-student" data-id="${
              application.id
            }">image</i>
            <span style="font-size: 0.75rem;">Attachment</span>
        </div>
        ${
          application.Passport_FPage
            ? `
        <div style="display: flex; align-items: center; gap: 6px; cursor: pointer;" title="Download Passport FPage">
            <i class="material-symbols-rounded text-black download-icon passport-fpage" data-id="${application.id}">badge</i>
            <span style="font-size: 0.75rem;">Passport</span>
        </div>
        `
            : `
        <div style="display: flex; align-items: center; gap: 6px; cursor: default; opacity: 0.5;" title="Passport not available">
            <i class="material-symbols-rounded text-black download-icon" style="cursor: not-allowed;">badge</i>
            <span style="font-size: 0.75rem;">Passport</span>
        </div>
        `
        }
    </td>
</tr>`;

    tbody.innerHTML += row;
  });

  const info = document.getElementById("pagination-info");
  const total = filteredApplications.length;
  const showingStart = total === 0 ? 0 : start + 1;
  const showingEnd = Math.min(end, total);
  info.textContent = `Showing ${showingStart}–${showingEnd} of ${total}`;

  document.getElementById("prevPage").disabled = page === 1;
  document.getElementById("nextPage").disabled = end >= total;

  const prevPageButton = document.getElementById("prevPage");
  const nextPageButton = document.getElementById("nextPage");

  prevPageButton.removeEventListener("click", handlePrevPageClick);
  nextPageButton.removeEventListener("click", handleNextPageClick);

  prevPageButton.addEventListener("click", handlePrevPageClick);
  nextPageButton.addEventListener("click", handleNextPageClick);
}

function handlePrevPageClick() {
  if (currentPage > 1) {
    currentPage--;
    renderPage(currentPage);
  }
}

function handleNextPageClick() {
  const statusFilter = document
    .getElementById("statusFilter")
    .value.toLowerCase();
  const searchInput = document
    .getElementById("searchInput")
    .value.toLowerCase()
    .trim();

  const filteredApplications = allApplications.filter((application) => {
    const rowStatus = (application.status ?? "new").toLowerCase();
    const studentName = (application.full_name ?? "").toLowerCase();

    const statusMatch = statusFilter === "all" || rowStatus === statusFilter;
    const nameMatch = searchInput === "" || studentName.includes(searchInput);

    return statusMatch && nameMatch;
  });

  const totalPages = Math.ceil(filteredApplications.length / pageSize);
  if (currentPage < totalPages) {
    currentPage++;
    renderPage(currentPage);
  }
}

// New function for file upload, to be used by all upload handlers
async function uploadFile(file, folderName) {
  try {
    const uploadPath = `applications/${folderName}/${file.name}`;
    const { error } = await window.supabase.storage
      .from("application-drive")
      .upload(uploadPath, file);

    if (error) {
      throw error;
    }

    const { publicUrl } = supabase.storage
      .from("application-drive")
      .getPublicUrl(uploadPath).data;

    return publicUrl;
  } catch (error) {
    console.error("File upload failed:", error);
    alert("Error uploading file: " + error.message);
    return null;
  }
}

async function handleFileUpload(studentId) {
  const student = allApplications.find((app) => app.id === studentId);
  if (!student) {
    alert("Student not found.");
    return;
  }

  const studentFolderName = student.full_name
    ? student.full_name.replace(/\s+/g, "_")
    : `student_${studentId}`;

  const emgsFile = document.getElementById("upload2-" + studentId)?.files[0];
  const evalFile = document.getElementById("upload3-" + studentId)?.files[0];

  let emgsUrl = null;
  if (emgsFile) {
    emgsUrl = await uploadFile(emgsFile, studentFolderName);
    if (!emgsUrl) return; // Exit if upload fails
  }

  let evalUrl = null;
  if (evalFile) {
    evalUrl = await uploadFile(evalFile, studentFolderName);
    if (!evalUrl) return; // Exit if upload fails
  }

  const updateData = {};
  if (emgsUrl) updateData.emgs_url = emgsUrl;
  if (evalUrl) updateData.eval_url = evalUrl;
  if (emgsUrl || evalUrl) {
    updateData.finalStatus = true;
  }

  const { error } = await window.supabase
    .from("student_applications")
    .update(updateData)
    .eq("id", studentId);

  if (error) {
    console.error("Error updating:", error.message);
    alert("Failed to update: " + error.message);
  } else {
    alert("Updated successfully.");
    location.reload();
  }
}

async function uploadZip(studentId) {
  const student = allApplications.find((app) => app.id === studentId);
  if (!student) {
    alert("Student not found.");
    return;
  }

  const studentFolderName = student.full_name
    ? student.full_name.replace(/\s+/g, "_")
    : `student_${studentId}`;
  const fileInput = document.getElementById("upload-zip-" + studentId);
  const file = fileInput?.files[0];

  if (!file) return alert("Please select a ZIP file");

  const offerLetterUrl = await uploadFile(file, studentFolderName);
  if (!offerLetterUrl) return;

  const { error } = await window.supabase
    .from("student_applications")
    .update({
      offer_letter_url: offerLetterUrl,
      OfferLetterStatus: true,
    })
    .eq("id", studentId);

  if (error) {
    console.error("Upload error:", error);
    alert("Upload failed");
  } else {
    alert("File uploaded successfully");
    location.reload();
  }
}

async function getReceiptDetail(studentId) {
  const { data, error } = await window.supabase
    .from("student_applications")
    .select("second_payment")
    .eq("id", studentId)
    .single();

  if (error) {
    console.error("Error fetching receipt photo:", error.message);
    alert("\u274C Error fetching receipt photo.");
    return null;
  }
  return data;
}

async function getStudentDetails(studentId) {
  try {
    if (!studentId) {
      alert("Invalid student ID");
      return null;
    }

    console.log("Fetching student details for ID:", studentId);

    const { data, error } = await window.supabase
      .from("student_applications")
      .select(
        `
                full_name, gender, date_of_birth, nationality, nonMalaysianInput,
                race, religion, ic_or_passport, email, birth_place, contact_number,
                emergency_contact_name, emergency_contact_number,
                permanent_address, permanent_postcode, permanent_city, permanent_state,
                program_status, program_name, study_mode, intake_year, intake_semester,
                highest_qualification, institution_name, institution_country, year_gradschool, graduation_year,
                secondary_name, secondary_address, secondary_qualification,
                english_test, english_test_date, english_score, english_test_other,
                require_visa, visa_type, passport_expiry, specifycondition, specifycurrent, current_medication, medicalConditions, father_name, father_occupation, father_phone, mother_name, mother_occupation, mother_phone, declaration_date, declaration_signature, declaration_checked,
                offer_letter_url, emgs_url, eval_url
                `
      )
      .eq("id", studentId)
      .single();

    if (error) {
      console.error("❌ Supabase error fetching student details:", error);
      alert("Error fetching student details.");
      return null;
    }

    if (!data) {
      alert("No student found with the given ID.");
      return null;
    }

    console.log("Fetched student details:", data);
    return data;
  } catch (err) {
    console.error("Unexpected error:", err);
    alert("Unexpected error fetching student details.");
    return null;
  }
}

async function getImageDetail(studentId) {
  const { data, error } = await window.supabase
    .from("student_applications")
    .select(
      `
            full_name,
            profile_photo,
            ResultQualification,
            Certificate_completion_studies_file,
            chsi,
            cscse,
            pre_chsi,
            pre_cscse,
            english_result_file,
            payment_receipt_file,
            graduatesecondry,
            Cert_sec_stud,
            Healthdeclare_file,
            parents_income
            `
    )
    .eq("id", studentId)
    .single();

  if (error) {
    console.error("❌ Error fetching attachment files:", error.message);
    alert("Failed to fetch attachments.");
    return null;
  }
  return data;
}

// Function to download passport PDF from URL
async function downloadPassportPDF(studentId) {
  try {
    const { data, error } = await window.supabase
      .from("student_applications")
      .select("Passport_FPage, full_name")
      .eq("id", studentId)
      .single();

    if (error) throw error;

    if (!data || !data.Passport_FPage) {
      alert("❗ No passport file found for this student.");
      return;
    }

    const passportUrl = data.Passport_FPage;
    const fullNameSafe = (data.full_name || "student").replace(/\s+/g, "_");

    const response = await fetch(passportUrl);
    if (!response.ok) throw new Error("Network response was not ok.");

    const blob = await response.blob();
    const link = document.createElement("a");
    link.href = URL.createObjectURL(blob);
    link.download = `Passport_${fullNameSafe}.pdf`;
    document.body.appendChild(link);
    link.click();
    document.body.removeChild(link);
  } catch (err) {
    console.error("❌ Failed to download Passport PDF:", err);
    alert(
      "❌ Failed to download passport. Please check if file exists and is a valid PDF."
    );
  }
}

// Function to generate and download a ZIP file of attachments
async function generateCombinedAttachmentsZip(profile) {
  const zip = new JSZip();

  const attachments = [
    {
      key: "profile_photo",
      label: "passport_photo",
    },
    {
      key: "ResultQualification",
      label: "academic_transcript",
    },
    {
      key: "Certificate_completion_studies_file",
      label: "academic_graduation_results",
    },
    {
      key: "chsi",
      label: "chsi",
    },
    {
      key: "cscse",
      label: "cscse",
    },
    {
      key: "pre_chsi",
      label: "pre_chsi",
    },
    {
      key: "pre_cscse",
      label: "pre_cscse",
    },
    {
      key: "english_result_file",
      label: "english_result",
    },
    {
      key: "payment_receipt_file",
      label: "payment_receipt",
    },
    {
      key: "graduatesecondry",
      label: "graduate_secondary_cert",
    },
    {
      key: "Cert_sec_stud",
      label: "certificate_secondary_studies",
    },
    {
      key: "Healthdeclare_file",
      label: "health_declaration",
    },
    {
      key: "parents_income",
      label: "parents_income",
    },
  ];

  let addedFileCount = 0;

  for (const { key, label } of attachments) {
    const fileUrl = profile[key];
    if (fileUrl) {
      try {
        const response = await fetch(fileUrl);
        if (!response.ok) {
          console.warn(`Failed to fetch file from URL: ${fileUrl}`);
          continue;
        }

        const blob = await response.blob();
        const urlParts = fileUrl.split("/");
        const fileNameWithQuery = urlParts[urlParts.length - 1];
        const fileName = fileNameWithQuery.split("?")[0];

        zip.file(fileName, blob);
        addedFileCount++;
      } catch (err) {
        console.error(`Error processing file from URL: ${fileUrl}`, err);
      }
    }
  }

  if (addedFileCount === 0) {
    alert("⚠️ No valid files found to zip.");
    return;
  }

  const content = await zip.generateAsync({
    type: "blob",
  });
  const link = document.createElement("a");
  const fullNameSafe = profile.full_name
    ? profile.full_name.replace(/\s+/g, "_")
    : "student";
  link.href = URL.createObjectURL(content);
  link.download = `${fullNameSafe}_attachments.zip`;
  document.body.appendChild(link);
  link.click();
  document.body.removeChild(link);
}

function generatePDF(student) {
  const { jsPDF } = window.jspdf;
  const doc = new jsPDF();

  const safe = (val) => val || "-";
  const formatDate = (val) =>
    val ? new Date(val).toLocaleDateString("en-GB") : "-";

  doc.setFontSize(14);
  doc.text("STUDENT APPLICATION DETAILS OF UNIMAP@KL", 14, 15);
  doc.setFontSize(10);

  doc.autoTable({
    startY: 25,
    head: [["SECTION", "FIELD", "DETAIL"]],
    body: [
      [
        {
          content: "PERSONAL INFORMATION",
          rowSpan: 11,
          styles: {
            fontStyle: "bold",
          },
        },
        "Full Name",
        safe(student.full_name),
      ],
      ["Gender", safe(student.gender)],
      ["Date of Birth", formatDate(student.date_of_birth)],
      [
        "Nationality",
        student.nationality?.toLowerCase?.() === "nonmalaysian"
          ? "Non-Malaysian"
          : student.nationality
          ? "Malaysian"
          : "Not specified",
      ],
      ["Specify Nationality", safe(student.nonMalaysianInput)],
      ["Race", safe(student.race)],
      ["Religion", safe(student.religion)],
      ["IC/Passport No.", safe(student.ic_or_passport)],
      ["Email", safe(student.email)],
      ["Place of Birth", safe(student.birth_place)],
      ["Contact Number", safe(student.contact_number)],
      [
        {
          content: "EMERGENCY INFORMATION",
          rowSpan: 1,
          styles: {
            fontStyle: "bold",
          },
        },
        "Emergency Contact",
        `${safe(student.emergency_contact_name)}, ${safe(
          student.emergency_contact_number
        )}`,
      ],
      [
        {
          content: "ADDRESS INFORMATION",
          rowSpan: 1,
          styles: {
            fontStyle: "bold",
          },
        },
        "Address",
        `${safe(student.permanent_address)}, ${safe(
          student.permanent_postcode
        )}, ${safe(student.permanent_city)}, ${safe(student.permanent_state)}`,
      ],
      [
        {
          content: "PROGRAM INFORMATION",
          rowSpan: 5,
          styles: {
            fontStyle: "bold",
          },
        },
        "Program Level",
        safe(student.program_status),
      ],
      ["Program Name", safe(student.program_name)],
      ["Mode of Study", safe(student.study_mode)],
      ["Intake Year", safe(student.intake_year)],
      ["Semester", safe(student.intake_semester)],
      [
        {
          content: "ACADEMIC INFORMATION",
          rowSpan: 4,
          styles: {
            fontStyle: "bold",
          },
        },
        "Highest Qualification Obtained",
        safe(student.highest_qualification),
      ],
      ["Institution Name", safe(student.institution_name)],
      ["Country of Institution", safe(student.institution_country)],
      ["Year of Graduation", safe(student.graduation_year)],
      [
        {
          content: "Supporting Education (Bachelor's for PhD Applicants only)",
          rowSpan: 3,
          styles: {
            fontStyle: "bold",
          },
        },
        "Secondary School Name",
        safe(student.secondary_name),
      ],
      ["Address", safe(student.secondary_address)],
      ["Qualifications Obtained", safe(student.secondary_qualification)],
      [
        {
          content: "ENGLISH PROFICIENCY",
          rowSpan: 3,
          styles: {
            fontStyle: "bold",
          },
        },
        "Type of English Test Taken",
        safe(student.english_test),
      ],
      ["Test Date", formatDate(student.english_test_date)],
      ["Score", safe(student.english_score)],
      [
        {
          content: "VISA INFORMATION",
          rowSpan: 3,
          styles: {
            fontStyle: "bold",
          },
        },
        "Require Student Visa?",
        safe(student.require_visa),
      ],
      ["Type of Visa", safe(student.visa_type)],
      ["Passport Expiry Date", formatDate(student.passport_expiry)],
      [
        {
          content: "HEALTH DECLARATION",
          rowSpan: 4,
          styles: {
            fontStyle: "bold",
          },
        },
        "Any known medical conditions?",
        safe(student.medicalConditions),
      ],
      ["If Yes, Specify", safe(student.specifycondition)],
      [
        "Currently on any regular medication?",
        safe(student.current_medication),
      ],
      ["If Yes, Specify", safe(student.specifycurrent)],
      [
        {
          content: "PARENT/GUARDIAN INFORMATION",
          rowSpan: 6,
          styles: {
            fontStyle: "bold",
          },
        },
        "Father's/Guardian's Name",
        safe(student.father_name),
      ],
      ["Father's/Guardian's Contact", safe(student.father_phone)],
      ["Father's/Guardian's Occupation", safe(student.father_occupation)],
      ["Mother's/Guardian's Name", safe(student.mother_name)],
      ["Mother's/Guardian's Contact", safe(student.mother_phone)],
      ["Mother's/Guardian's Occupation", safe(student.mother_occupation)],
      [
        {
          content: "DECLARATION",
          rowSpan: 4,
          styles: {
            fontStyle: "bold",
          },
        },
        "Declaration",
        "I declare all the information provided is accurate.",
      ],
      ["Declare: Yes/No", safe(student.declaration_checked ? "Yes" : "No")],
      ["Signature of Student", safe(student.declaration_signature)],
      ["Declaration Date", formatDate(student.declaration_date)],
    ],
    styles: {
      fontSize: 10,
      cellWidth: "wrap",
    },
    columnStyles: {
      0: {
        cellWidth: 50,
      },
      1: {
        cellWidth: 50,
      },
      2: {
        cellWidth: 90,
      },
    },
    headStyles: {
      fillColor: [0, 51, 102],
      textColor: [255, 255, 255],
      halign: "center",
    },
    theme: "grid",
  });

  doc.save(`${safe(student.full_name).replace(/\s+/g, "_")}_Application.pdf`);
}

function generatePDFReceipt(photo) {
  const { jsPDF } = window.jspdf;
  const doc = new jsPDF();

  const receiptData = photo.second_payment;

  if (!receiptData) {
    alert("❗ No receipt data available.");
    return;
  }

  if (receiptData.startsWith("data:application/pdf")) {
    const byteCharacters = atob(receiptData.split(",")[1]);
    const byteNumbers = new Array(byteCharacters.length)
      .fill()
      .map((_, i) => byteCharacters.charCodeAt(i));
    const byteArray = new Uint8Array(byteNumbers);
    const blob = new Blob([byteArray], {
      type: "application/pdf",
    });

    const link = document.createElement("a");
    link.href = URL.createObjectURL(blob);
    link.download = "Receipt.pdf";
    document.body.appendChild(link);
    link.click();
    document.body.removeChild(link);

    return;
  }

  const pageWidth = doc.internal.pageSize.getWidth();
  const pageHeight = doc.internal.pageSize.getHeight();

  doc.setFontSize(12);
  doc.text("Second Payment Receipt", 10, 10);
  doc.addImage(receiptData, "JPEG", 0, 20, pageWidth, pageHeight - 20);

  doc.save("Receipt_Image.pdf");
}

const nationalitySelect = document.getElementById("nationality");
const nonMalaysianInput = document.getElementById("nonMalaysianInput");
if (nationalitySelect && nonMalaysianInput) {
  nationalitySelect.addEventListener("change", function () {
    if (this.value === "NonMalaysian") {
      nonMalaysianInput.style.display = "block";
    } else {
      nonMalaysianInput.style.display = "none";
    }
  });
}

const englishTestSelect = document.getElementById("english_test");
const englishTestInput = document.getElementById("english_test_other");
if (englishTestSelect && englishTestInput) {
  englishTestSelect.addEventListener("change", function () {
    if (this.value === "Other") {
      englishTestInput.style.display = "block";
    } else {
      englishTestInput.style.display = "none";
    }
  });
}

document.addEventListener("DOMContentLoaded", fetchApplicationStudent);
