// migrate_fix_folder_structure.js
import { resolve, dirname } from "path";
import { fileURLToPath } from "url";
import dotenv from "dotenv";
import { createClient } from "@supabase/supabase-js";
import pg from "pg";
import fetch from "node-fetch"; // npm install node-fetch@2 jika Node <18

const { Pool } = pg;
const __dirname = dirname(fileURLToPath(import.meta.url));
dotenv.config({ path: resolve(__dirname, "../../../.env") });

// Cek environment variables
console.log("[ENV CHECK]");
[
  "NEXT_PUBLIC_SUPABASE_URL",
  "SUPABASE_SERVICE_ROLE_KEY",
  "DATABASE_URL",
].forEach((key) => {
  console.log(`${key}: ${process.env[key] ? "✅ OK" : "❌ MISSING"}`);
});
if (
  !process.env.NEXT_PUBLIC_SUPABASE_URL ||
  !process.env.SUPABASE_SERVICE_ROLE_KEY ||
  !process.env.DATABASE_URL
) {
  throw new Error("❌ Missing environment variables. Please check .env");
}

const SUPABASE_URL = process.env.NEXT_PUBLIC_SUPABASE_URL;
const SUPABASE_SERVICE_KEY = process.env.SUPABASE_SERVICE_ROLE_KEY;
const BUCKET_NAME = "application-drive";
const supabase = createClient(SUPABASE_URL, SUPABASE_SERVICE_KEY);
const pool = new Pool({ connectionString: process.env.DATABASE_URL });

// Senarai kolom fail dalam table
const columnsToMigrate = [
  "profile_photo",
  "ResultQualification",
  "english_result_file",
  "Passport_FPage",
  "Healthdeclare_file",
  "payment_receipt_file",
  "declaration_signature",
  "graduatesecondry",
  "Cert_sec_stud",
  "Certificate_completion_studies_file",
  "second_payment",
  "chsi",
  "cscse",
  "pre_chsi",
  "pre_cscse",
  "eval_url",
  "offer_letter_url",
  "emgs_url",
];

// Fungsi untuk sanitize nama folder supaya tiada karakter pelik
function sanitizeFolderName(name) {
  return name
    .toString()
    .trim()
    .replace(/[^\w\-]/g, "_");
}

// Fungsi untuk dapatkan ekstensi fail dari URL (fallback ke bin)
function getExtFromUrl(url) {
  const parts = url.split(".").pop().split(/\#|\?/);
  return parts[0] || "bin";
}

async function migrateFolderStructure() {
  const client = await pool.connect();
  let successCount = 0;
  let failCount = 0;
  let skipCount = 0;

  try {
    const { rows } = await client.query(`
      SELECT id, user_id, user_email, ${columnsToMigrate.join(", ")}
      FROM student_applications
    `);

    console.log(`[INFO] Found ${rows.length} records.`);

    for (const row of rows) {
      // Folder ikut user_id atau user_email jika user_id kosong
      const userFolder = sanitizeFolderName(
        row.user_id || row.user_email || "unknown_user"
      );
      const applicationId = row.id.toString().trim();

      for (const col of columnsToMigrate) {
        const oldUrl = row[col];
        // Pastikan ada URL fail dan valid HTTP(S)
        if (
          !oldUrl ||
          typeof oldUrl !== "string" ||
          !oldUrl.startsWith("http")
        ) {
          skipCount++;
          continue;
        }

        try {
          // Download fail dari URL lama
          const res = await fetch(oldUrl);
          if (!res.ok)
            throw new Error(
              `Failed to download file: ${res.status} ${res.statusText}`
            );

          const buffer = await res.buffer();
          const ext = getExtFromUrl(oldUrl).toLowerCase();

          // Susun path fail ikut folder user/application/column.ext
          const filePath = `${userFolder}/${applicationId}/${col}.${ext}`;

          // Upload semula fail ke Supabase Storage dengan upsert true
          const { error: uploadError } = await supabase.storage
            .from(BUCKET_NAME)
            .upload(filePath, buffer, {
              contentType: res.headers.get("content-type") || undefined,
              upsert: true,
            });

          if (uploadError) {
            console.error(
              `❌ Upload failed for ${filePath}:`,
              uploadError.message
            );
            failCount++;
            continue;
          }

          // Dapatkan public URL baru
          const { data: publicUrlData } = supabase.storage
            .from(BUCKET_NAME)
            .getPublicUrl(filePath);

          // Update URL baru dalam database
          await client.query(
            `UPDATE student_applications SET ${col} = $1 WHERE id = $2`,
            [publicUrlData.publicUrl, row.id]
          );

          console.log(`✅ Migrated ${col} for record ${row.id} to ${filePath}`);
          successCount++;
        } catch (err) {
          console.error(
            `❌ Error migrating ${col} for record ${row.id}:`,
            err.message
          );
          failCount++;
        }
      }
    }

    console.log("\n=== Migration Summary ===");
    console.log(`✅ Success: ${successCount}`);
    console.log(`❌ Fail: ${failCount}`);
    console.log(`⚠️ Skipped: ${skipCount}`);
    console.log("=========================");
  } finally {
    client.release();
  }
}

migrateFolderStructure()
  .then(() => {
    console.log("Folder restructure migration complete.");
    process.exit(0);
  })
  .catch((err) => {
    console.error("Migration failed:", err);
    process.exit(1);
  });
